﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;

namespace JHMA_Console
{
    static class Program
    {
        public static UIDataSet ForLang = new UIDataSet();
        public static string GetLang(string name)
        {
            UIDataSet.LangRow lrow = ForLang.Lang.FindByName(name);
            if (lrow != null)
            {
                return lrow.Display;
            }
            return name;
        }

        /// <summary>
        /// 应用程序的主入口点。
        /// </summary>
        [STAThread]

        static void Main()
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            string comboName = Application.StartupPath + @"\JHMA_Console.xml";
            ForLang.Lang.ReadXml(comboName);

            Application.Run(new MainForm());
        }
    }

    class Aes // Aes类
    {
        public enum KeySize { Bits128, Bits192, Bits256 };//密钥长度指定为128，192，256位，适合用枚举类型表示
        private int Nb;//以32bits位（字）为单位的明文每一个分组包含的块数，在AES中（明文分组为128bits）恒为4
        private int Nk;//以32bits位（字）为单位的密钥每一个分组包含的块数，在AES中为4(128bits)，6(128bits)，8(256bits)
        private int Nr;//轮数，为10，12，14，取决于密钥长度
        private byte[] key;//原始密钥数组，密钥数组长度为4×Nk
        private byte[,] Sbox;//S盒，为16×16字节矩阵
        private byte[,] iSbox;//逆S盒，为16×16字节矩阵
        private byte[,] w;//扩展密钥表，44个字，每个字32bits，即byte[44,4] w
        private byte[,] Rcon;//轮常数表，为byte[Nr,4] Rcon
        private byte[,] State;//State数组，为4×4字节矩阵，即byte[4,4] State
        //以下为Aes类初始化部分
        /********************************************************************************************************************************************/
        //Aes构造函数，用于初始化Nb,Nk,Nr,key,Sbox,iSbox
        public Aes(KeySize keySize, byte[] keyBytes)
        {
            SetNbNkNr(keySize);//根据密钥长度keySize设置Nb,Nk,Nr
            this.key = new byte[this.Nk * 4];//根据Nk设置密钥数组key长度
            keyBytes.CopyTo(this.key, 0);//给key数组赋值
            BuildSbox();//初始化“Sbox”
            BuildInvSbox();//初始化“逆Sbox”
            BuildRcon();//初始化轮常数表Rcon
            KeyExpansion();//密钥扩展
        }
        //SetNbNkNr()函数，根据密钥长度初始化Nb，Nk，Nr
        private void SetNbNkNr(KeySize keySize)
        {
            this.Nb = 4;//Nb恒为4，AES明文分组为128bits，以32bits（字）为单位分块的话，块数Nb就是4
            if (keySize == KeySize.Bits128)//密钥长度为128bits
            {
                this.Nk = 4;//32bits×4＝128bits
                this.Nr = 10;
            }
            if (keySize == KeySize.Bits192)//密钥长度为192bits
            {
                this.Nk = 6;//32bits×6＝192bits
                this.Nr = 12;
            }
            if (keySize == KeySize.Bits256)//密钥长度为256bits
            {
                this.Nk = 8;//32bits×8＝256bits
                this.Nr = 14;
            }
        }
        //BuildSbox()函数，初始化S盒，见课本P106表5.4（a）S盒
        private void BuildSbox()
        {
            this.Sbox = new byte[16, 16]{//给S盒分配空间，为16×16字节矩阵,并赋值
          /* 0     1     2     3     4     5     6     7     8     9     a     b     c     d     e     f */
    /*0*/  {0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76},
    /*1*/  {0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0},
    /*2*/  {0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15},
    /*3*/  {0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75},
    /*4*/  {0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84},
    /*5*/  {0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf},
    /*6*/  {0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8},
    /*7*/  {0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2},
    /*8*/  {0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73},
    /*9*/  {0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb},
    /*a*/  {0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79},
    /*b*/  {0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08},
    /*c*/  {0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a},
    /*d*/  {0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e},
    /*e*/  {0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf},
    /*f*/  {0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16} };

        }
        //BuildInvSbox()函数，初始化逆S盒，见课本P106表5.4（b）逆S盒
        private void BuildInvSbox()
        {
            this.iSbox = new byte[16, 16]{ //给逆S盒分配空间，为16×16字节矩阵,并赋值
          /* 0     1     2     3     4     5     6     7     8     9     a     b     c     d     e     f */
    /*0*/  {0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb},
    /*1*/  {0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb},
    /*2*/  {0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e},
    /*3*/  {0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25},
    /*4*/  {0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92},
    /*5*/  {0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84},
    /*6*/  {0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06},
    /*7*/  {0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b},
    /*8*/  {0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73},
    /*9*/  {0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e},
    /*a*/  {0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b},
    /*b*/  {0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4},
    /*c*/  {0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f},
    /*d*/  {0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef},
    /*e*/  {0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61},
    /*f*/  {0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d} };
        }
        //BuildRcon()函数，初始化轮常数表Rcon，见课本P111上面的轮常数表
        private void BuildRcon()
        {
            this.Rcon = new byte[11, 4] { 
                                   {0x00, 0x00, 0x00, 0x00},  
                                   {0x01, 0x00, 0x00, 0x00},
                                   {0x02, 0x00, 0x00, 0x00},
                                   {0x04, 0x00, 0x00, 0x00},
                                   {0x08, 0x00, 0x00, 0x00},
                                   {0x10, 0x00, 0x00, 0x00},
                                   {0x20, 0x00, 0x00, 0x00},
                                   {0x40, 0x00, 0x00, 0x00},
                                   {0x80, 0x00, 0x00, 0x00},
                                   {0x1b, 0x00, 0x00, 0x00},
                                   {0x36, 0x00, 0x00, 0x00} };
        }
        //KeyExpansion()函数，进行密钥扩展，参照课本P110的伪代码
        private void KeyExpansion()
        {
            //首先，输入密钥直接被复制到扩展密钥数组的前Nk(4-128,6-192,8-256)个字。
            this.w = new byte[Nb * (Nr + 1), 4];   //（1＋轮数Nr）×128bits的线性数组w   
            for (int row = 0; row < Nk; ++row)
            {
                this.w[row, 0] = this.key[4 * row];
                this.w[row, 1] = this.key[4 * row + 1];
                this.w[row, 2] = this.key[4 * row + 2];
                this.w[row, 3] = this.key[4 * row + 3];
            }
            //然后，每次用4个字 / 128bits填充扩展密钥数组余下的部分。在扩展密钥数组中，w[i]的值依赖于w[i-1]和w[i-Nk].
            byte[] temp = new byte[4];//一个temp对应一个w,因为在C#中没有 word类型，用 4字节数组实现了一个字。
            for (int row = Nk; row < Nb * (Nr + 1); ++row)//Nk为密码块数
            {
                //对应课本P110的temp=w[i-1]
                temp[0] = this.w[row - 1, 0]; temp[1] = this.w[row - 1, 1];
                temp[2] = this.w[row - 1, 2]; temp[3] = this.w[row - 1, 3];
                if (row % Nk == 0)
                {
                    //对应课本P110的temp=SubWord(RotWord(temp))XOR Rcon[i/Nk]
                    temp = SubWord(RotWord(temp));
                    //两个字节相互的异或操作。它需要一些从 byte 到 int 的强制类型转换并转回到 byte，因为异或操作“^”是不能定义在 C# 的 byte 类型上
                    temp[0] = (byte)((int)temp[0] ^ (int)this.Rcon[row / Nk, 0]);
                    temp[1] = (byte)((int)temp[1] ^ (int)this.Rcon[row / Nk, 1]);
                    temp[2] = (byte)((int)temp[2] ^ (int)this.Rcon[row / Nk, 2]);
                    temp[3] = (byte)((int)temp[3] ^ (int)this.Rcon[row / Nk, 3]);
                }
                else if (Nk > 6 && (row % Nk == 4))
                {
                    temp = SubWord(temp);
                }
                // 对应课本P110的w[i] = w[i-Nk] xor temp
                this.w[row, 0] = (byte)((int)this.w[row - Nk, 0] ^ (int)temp[0]);
                this.w[row, 1] = (byte)((int)this.w[row - Nk, 1] ^ (int)temp[1]);
                this.w[row, 2] = (byte)((int)this.w[row - Nk, 2] ^ (int)temp[2]);
                this.w[row, 3] = (byte)((int)this.w[row - Nk, 3] ^ (int)temp[3]);
            }
        }
        //函数RotWord()：实现字循环功能，使一个字的4个字节循环左移一个字节
        private byte[] RotWord(byte[] word)
        {
            byte[] result = new byte[4];
            result[0] = word[1];
            result[1] = word[2];
            result[2] = word[3];
            result[3] = word[0];
            return result;
        }
        //函数SubWord()：字节代换利用S盒对输入字中的每个字节进行字节代换
        private byte[] SubWord(byte[] word)
        {
            byte[] result = new byte[4];
            result[0] = this.Sbox[word[0] >> 4/*高4位*/, word[0] & 0x0f/*低4位*/];
            result[1] = this.Sbox[word[1] >> 4/*高4位*/, word[1] & 0x0f/*低4位*/];
            result[2] = this.Sbox[word[2] >> 4/*高4位*/, word[2] & 0x0f/*低4位*/];
            result[3] = this.Sbox[word[3] >> 4/*高4位*/, word[3] & 0x0f/*低4位*/];
            /*
            对于一个给定字节，用 >> 操作符右移 4 位将得到 x 索引，并且与 0000 1111 进行逻辑与得到 y 值。虽然有些长，但比实际代码更可读，可以象下面这样：
            int x = word[0] >> 4;
            int y = word[0] & 0x0f;
            byte substitute = this.Sbox[x,y];
            result[0] = substitute;
            代替上面用的代码：
            */
            return result;
        }
        /********************************************************************************************************************************************/





        //以下为加密过程        
        /********************************************************************************************************************************************/
        //加密函数
        public void Cipher(byte[] input, int offset, byte[] output)
        {
            //接收输入，存入State数组
            this.State = new byte[4, Nb];//Nb恒为4
            for (int i = 0; i < 16; i++)
            {
                this.State[i % 4, i / 4] = input[i + offset];//State数组竖着存放输入字符串
            }
            AddRoundKey(0);//初始轮密钥加，第0轮
            for (int round = 1; round <= (Nr - 1); round++)//每一轮过程如下
            {
                SubBytes();//字节代换
                ShiftRows();//行移位
                MixColumns();//列混淆
                AddRoundKey(round);//第round轮
            }
            //最后一轮，第Nr轮
            SubBytes();//字节代换
            ShiftRows();//行移位
            AddRoundKey(Nr);//第Nr轮
            //输出处理
            for (int i = 0; i < 16; i++)
            {
                output[i] = this.State[i % 4, i / 4];//State数组竖着赋值到输出字符串
            }

        }
        //AddRoundKey()轮密钥加函数
        private void AddRoundKey(int round)
        {
            for (int r = 0; r < 4; ++r)//r为列
            {
                for (int c = 0; c < 4; ++c)//c为行
                {
                    this.State[c, r] = (byte)((int)this.State[c, r] ^ (int)w[(round * 4) + r, c]);//XOR
                }
            }
        }

        //SubBytes()字节代换函数
        private void SubBytes()
        {
            for (int i = 0; i < 4; i++)
                for (int j = 0; j < 4; j++)
                {
                    this.State[j, i] = this.Sbox[(int)(this.State[j, i] >> 4)/*高4位*/, (int)(this.State[j, i] & 0x0f)/*低4位*/];
                }
        }
        //ShiftRows()行变换函数，第0行循环左移0字节，第1行循环左移1字节，第2行循环左移2字节，第3行循环左移3字节
        private void ShiftRows()
        {
            byte[,] temp = new byte[4, 4];
            for (int r = 0; r < 4; ++r)  // 复制State到临时数组temp[]
            {
                for (int c = 0; c < 4; ++c)
                {
                    temp[r, c] = this.State[r, c];
                }
            }
            for (int r = 0; r < 4; ++r)  //第0行循环左移0字节，第1行循环左移1字节，第2行循环左移2字节，第3行循环左移3字节
            {
                for (int c = 0; c < 4; ++c)
                {
                    this.State[r, c] = temp[r, (c + r) % Nb];//r表示循环左移的字节数
                }
            }
        }
        //MixColumns()列混淆函数
        private void MixColumns()
        {
            byte[,] temp = new byte[4, 4];
            for (int r = 0; r < 4; ++r)  // 复制 State 到临时数组 temp[]
            {
                for (int c = 0; c < 4; ++c)
                {
                    temp[r, c] = this.State[r, c];
                }
            }
            for (int c = 0; c < 4; ++c)//参照课本P107图5.5和式子5.4，注意运算都是在GF(2E8)上的
            {
                this.State[0, c] = (byte)((int)gfmultby02(temp[0, c]) ^ (int)gfmultby03(temp[1, c]) ^
                                           (int)gfmultby01(temp[2, c]) ^ (int)gfmultby01(temp[3, c]));
                this.State[1, c] = (byte)((int)gfmultby01(temp[0, c]) ^ (int)gfmultby02(temp[1, c]) ^
                                           (int)gfmultby03(temp[2, c]) ^ (int)gfmultby01(temp[3, c]));
                this.State[2, c] = (byte)((int)gfmultby01(temp[0, c]) ^ (int)gfmultby01(temp[1, c]) ^
                                           (int)gfmultby02(temp[2, c]) ^ (int)gfmultby03(temp[3, c]));
                this.State[3, c] = (byte)((int)gfmultby03(temp[0, c]) ^ (int)gfmultby01(temp[1, c]) ^
                                           (int)gfmultby01(temp[2, c]) ^ (int)gfmultby02(temp[3, c]));
            }
        }
        //GF(2E8)上的运算如下：
        private static byte gfmultby01(byte b)
        {
            return b;
        }
        private static byte gfmultby02(byte b)
        {
            if (b < 0x80)
                return (byte)(int)(b << 1);
            else
                return (byte)((int)(b << 1) ^ (int)(0x1b));
        }
        private static byte gfmultby03(byte b)
        {
            return (byte)((int)gfmultby02(b) ^ (int)b);
        }
        private static byte gfmultby09(byte b)
        {
            return (byte)((int)gfmultby02(gfmultby02(gfmultby02(b))) ^
                           (int)b);
        }
        private static byte gfmultby0b(byte b)
        {
            return (byte)((int)gfmultby02(gfmultby02(gfmultby02(b))) ^
                           (int)gfmultby02(b) ^
                           (int)b);
        }
        private static byte gfmultby0d(byte b)
        {
            return (byte)((int)gfmultby02(gfmultby02(gfmultby02(b))) ^
                           (int)gfmultby02(gfmultby02(b)) ^
                           (int)(b));
        }
        private static byte gfmultby0e(byte b)
        {
            return (byte)((int)gfmultby02(gfmultby02(gfmultby02(b))) ^
                           (int)gfmultby02(gfmultby02(b)) ^
                           (int)gfmultby02(b));
        }
        /********************************************************************************************************************************************/


        //以下为解密过程        
        /********************************************************************************************************************************************/
        //InvCipher()解密函数
        public void InvCipher(byte[] input, int offset, byte[] output)  // decipher 16-bit input
        {
            //接收输入，存入State数组
            this.State = new byte[4, Nb];//Nb恒为4
            for (int i = 0; i < 16; i++)
            {
                this.State[i % 4, i / 4] = input[i + offset];//State数组竖着存放输入字符串
            }
            AddRoundKey(Nr);//初始轮密钥加，第Nr轮
            for (int round = Nr - 1; round >= 1; round--)  //每一轮过程如下
            {
                InvShiftRows();//行移位求逆
                InvSubBytes();//字节代换求逆
                AddRoundKey(round);//轮密钥加
                InvMixColumns();//列混淆求逆
            }
            //最后一轮
            InvShiftRows();//行移位求逆
            InvSubBytes();//字节代换求逆
            AddRoundKey(0);//轮密钥加
            //输出处理
            for (int i = 0; i < (4 * Nb); ++i)
            {
                output[i] = this.State[i % 4, i / 4];//State数组竖着赋值到输出字符串
            }
        }
        //InvSubBytes()字节代换求逆函数
        private void InvSubBytes()
        {
            for (int i = 0; i < 4; i++)
                for (int j = 0; j < 4; j++)
                {
                    this.State[j, i] = this.iSbox[(int)(this.State[j, i] >> 4)/*高4位*/, (int)(this.State[j, i] & 0x0f)/*低4位*/];
                }
        }
        //InvShiftRows()行变换求逆函数，第0行循环右移0字节，第1行循环右移1字节，第2行循环右移2字节，第3行循环右移3字节
        private void InvShiftRows()
        {
            byte[,] temp = new byte[4, 4];
            for (int r = 0; r < 4; ++r)  // 复制State到临时数组temp[]
            {
                for (int c = 0; c < 4; ++c)
                {
                    temp[r, c] = this.State[r, c];
                }
            }
            for (int r = 0; r < 4; ++r)  //第0行循环右移0字节，第1行循环右移1字节，第2行循环右移2字节，第3行循环右移3字节
            {
                for (int c = 0; c < 4; ++c)
                {
                    this.State[r, (c + r) % Nb] = temp[r, c];//这里只要与循环左移的下标对调即可
                }
            }
        }
        //InvMixColumns()列混淆求逆函数
        private void InvMixColumns()
        {
            byte[,] temp = new byte[4, 4];
            for (int r = 0; r < 4; ++r)  //复制 State 到临时数组 temp[]
            {
                for (int c = 0; c < 4; ++c)
                {
                    temp[r, c] = this.State[r, c];
                }
            }
            for (int c = 0; c < 4; ++c)//参照课本P108式子5.5，注意运算都是在GF(2E8)上的，不过这里使用了
            //一个别人写的方法，使用相同的矩阵，而课本中的矩阵使用已包含在gfmultby函数之中
            {
                this.State[0, c] = (byte)((int)gfmultby0e(temp[0, c]) ^ (int)gfmultby0b(temp[1, c]) ^
                                           (int)gfmultby0d(temp[2, c]) ^ (int)gfmultby09(temp[3, c]));
                this.State[1, c] = (byte)((int)gfmultby09(temp[0, c]) ^ (int)gfmultby0e(temp[1, c]) ^
                                           (int)gfmultby0b(temp[2, c]) ^ (int)gfmultby0d(temp[3, c]));
                this.State[2, c] = (byte)((int)gfmultby0d(temp[0, c]) ^ (int)gfmultby09(temp[1, c]) ^
                                           (int)gfmultby0e(temp[2, c]) ^ (int)gfmultby0b(temp[3, c]));
                this.State[3, c] = (byte)((int)gfmultby0b(temp[0, c]) ^ (int)gfmultby0d(temp[1, c]) ^
                                           (int)gfmultby09(temp[2, c]) ^ (int)gfmultby0e(temp[3, c]));
            }
        }
    }

}
